<?php

namespace App\Controllers;

use App\Models\ResearchJob;
use App\Services\AI\GeminiClient;
use App\Services\Agents\DiscoveryAgent;
use App\Services\Agents\PlannerAgent;
use App\Services\Agents\ResearcherAgent;
use App\Services\Agents\WriterAgent;
use Exception;

class StreamController
{
    /**
     * تنظیم مدل‌های پیشرفته (Generative AI):
     * برای سرعت از 2.5 و برای هوش از 3 استفاده می‌کنیم.
     */
    const MODEL_FAST = 'gemini-2.5-flash';
    const MODEL_SMART = 'gemini-3-pro-preview';

    public function handle()
    {
        // 1. تنظیمات سرور برای استریم زنده (No-Buffer)
        $this->initServer();

        // 2. اعتبارسنجی
        $id = $_GET['id'] ?? null;
        if (!$id) {
            $this->sendFatalError('شناسه پژوهش یافت نشد.');
            return;
        }

        $job = ResearchJob::find($id);
        if (!$job) {
            $this->sendFatalError("پژوهش مورد نظر در سیستم وجود ندارد.");
            return;
        }

        $this->sendEvent('log', "System Connected. Engine: " . self::MODEL_SMART);

        try {
            $topic = $job['user_topic'];
            
            // دریافت و بازگشایی پارامترها
            $params = json_decode($job['params'] ?? '{}', true);
            $approvedSources = $params['approved_sources'] ?? []; 

            // === بسته بندی کامل داده‌ها برای ارسال به WriterAgent ===
            // تمام فیلد هایی که در سایدبار پر کردید اینجا مپ می‌شوند
            $context = [
                // تنظیمات عمومی
                'role'          => $params['role'] ?? 'نویسنده حرفه‌ای',
                'context'       => $params['context'] ?? '',
                'audience'      => $params['audience'] ?? 'General',
                'style'         => $params['type'] ?? 'Blog Post', // نوع محتوا (Type)
                'tone'          => $params['tone'] ?? 'Professional',
                'language'      => $params['language'] ?? 'Persian',
                'constraints'   => $params['constraints'] ?? '',
                'word_count'    => $params['word_count'] ?? '1500',
                
                // تنظیمات پیشرفته (سئو و برندینگ)
                'focus_keyword' => $params['focus_keyword'] ?? '',
                'lsi_keywords'  => $params['lsi_keywords'] ?? '',
                'brand_name'    => $params['brand_name'] ?? '',
                'cta'           => $params['cta'] ?? '',
                'internal_links'=> $params['internal_links'] ?? '',
                'include_faq'   => $params['include_faq'] ?? false,
                'pov'           => $params['pov'] ?? 'Third Person',
                'angle'         => $params['angle'] ?? 'Standard'
            ];

            // 3. راه‌اندازی ایجنت‌ها
            $this->sendEvent('progress', ['percent' => 5, 'message' => "Configuring AI Agents..."]);
            $client = new GeminiClient();
            
            $planner = new PlannerAgent($client);
            $researcher = new ResearcherAgent($client);
            $writer = new WriterAgent($client);


            // === FASE 1: PLANNING (استراتژی) ===
            ResearchJob::update($id, ['status' => 'planning']);
            $this->sendEvent('progress', ['percent' => 15, 'message' => "🧠 Creating content strategy (" . count($approvedSources) . " sources)..."]);
            
            // پلنر بر اساس منابعی که شما تیک زدید استراتژی می‌چیند
            $plan = $planner->generatePlan($topic, $context, self::MODEL_SMART, $approvedSources);
            
            ResearchJob::update($id, ['plan_data' => json_encode($plan)]);
            $this->sendEvent('log', "Plan Created: " . count($plan) . " key sections identified.");


            // === FASE 2: DEEP RESEARCH (جستجو و استخراج) ===
            ResearchJob::update($id, ['status' => 'researching']);
            $fullNotes = "";
            $totalQueries = count($plan);
            
            foreach ($plan as $idx => $query) {
                $percent = 20 + round((($idx + 1) / $totalQueries) * 65);
                $this->sendEvent('progress', ['percent' => $percent, 'message' => "🔎 Deep Diving: $query"]);
                $this->sendEvent('log', "Analyzing data points...");
                
                // تاخیر کوتاه برای جلوگیری از خطای ریت لیمیت
                sleep(2);

                try {
                    // جستجو برای فکت‌ها
                    $res = $client->generateContent(
                        "Conduct detailed research on: '$query'. Focus on finding specific statistics, technical details, and quotes suitable for a {$context['style']}.",
                        self::MODEL_FAST,
                        true,  // سرچ فعال
                        false  // متن خام می‌خواهیم نه جیسون
                    );
                    
                    $text = $res['candidates'][0]['content']['parts'][0]['text'] ?? 'Data not found.';
                    
                    // فرمت‌بندی دقیق برای بخش ارجاع‌دهی (Citations)
                    // عنوان کوئری را به عنوان رفرنس حدودی نگه می‌داریم
                    $fullNotes .= "### Source Context: Research on ['$query']\nContent: $text\n\n-----------------\n\n";
                    
                    $len = mb_strlen($text);
                    $this->sendEvent('log', "✔ Extracted {$len} chars of insight.");

                } catch (Exception $ex) {
                    $this->sendEvent('log', "⚠️ Skipped step: " . $ex->getMessage());
                }
            }
            
            ResearchJob::update($id, ['research_notes' => $fullNotes]);


            // === FASE 3: WRITING (نگارش نهایی) ===
            ResearchJob::update($id, ['status' => 'writing']);
            $this->sendEvent('progress', ['percent' => 95, 'message' => "✍️ Synthesizing Final Content..."]);
            
            $this->sendEvent('log', "Writing with " . self::MODEL_SMART);
            sleep(2);
            
            // ارسال تمام اطلاعات به نویسنده
            $finalReport = $writer->writeReport($topic, $fullNotes, $context, self::MODEL_SMART);
            
            ResearchJob::update($id, [
                'status' => 'completed',
                'final_report' => $finalReport
            ]);

            // ریدایرکت
            $this->sendEvent('complete', ['redirect' => 'report?id=' . $id]);

        } catch (Exception $e) {
            $this->sendFatalError($e->getMessage());
            ResearchJob::update($id, ['status' => 'failed']);
        }
        
        echo "\n\n";
        exit;
    }

    private function initServer()
    {
        if (function_exists('apache_setenv')) {
            @apache_setenv('no-gzip', 1);
        }
        @ini_set('zlib.output_compression', 0);
        @ini_set('implicit_flush', 1);
        while (ob_get_level() > 0) {
            ob_end_clean();
        }
        
        header('Content-Type: text/event-stream');
        header('Cache-Control: no-cache');
        header('Connection: keep-alive');
        header('X-Accel-Buffering: no'); 
        
        set_time_limit(0); 
    }

    private function sendEvent($event, $data) {
        echo "event: $event\n";
        echo "data: " . json_encode($data) . "\n\n";
        if (ob_get_level() > 0) ob_flush();
        flush();
    }
    
    private function sendFatalError($msg) {
        $this->sendEvent('log', "❌ ERROR: $msg");
        $this->sendEvent('error', ['message' => $msg]);
        if (ob_get_level() > 0) ob_flush();
        flush();
    }
}