<?php

namespace App\Services\AI;

use App\Core\Env;
use Exception;

class GeminiClient
{
    private string $apiKey;
    private string $baseUrl = "https://generativelanguage.googleapis.com/v1beta/models/";

    public function __construct()
    {
        $this->apiKey = Env::get('GEMINI_API_KEY');
        if (!$this->apiKey) throw new Exception("API Key missing.");
    }

    public function generateContent(string $prompt, string $model = 'gemini-1.5-flash', bool $enableSearch = false, bool $forceJson = false): array
    {
        $model = trim($model);
        
        $url = $this->baseUrl . $model . ":generateContent?key=" . $this->apiKey;

        // تنظیمات تولید محتوا
        $generationConfig = [
            'temperature' => 0.7,
            // تغییر ۱: افزایش سقف توکن به حداکثر ممکن (۸۱۹۲ برای اکثر مدل‌ها)
            'maxOutputTokens' => 8192 
        ];

        if ($forceJson) {
            $generationConfig['responseMimeType'] = 'application/json';
        }

        $payload = [
            'contents' => [['parts' => [['text' => $prompt]]]],
            'generationConfig' => $generationConfig
        ];

        if ($enableSearch) {
            $payload['tools'] = [['google_search' => new \stdClass()]];
        }

        return $this->requestWithRetry($url, $payload);
    }

    private function requestWithRetry(string $url, array $payload, int $maxRetries = 2): array
    {
        $attempt = 0;
        while ($attempt <= $maxRetries) {
            $attempt++;
            
            $ch = curl_init($url);
            curl_setopt_array($ch, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => json_encode($payload),
                CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
                // تغییر ۲: افزایش زمان انتظار سرور برای متن‌های طولانی (۳ دقیقه)
                CURLOPT_TIMEOUT => 180, 
                CURLOPT_CONNECTTIMEOUT => 30
            ]);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $err = curl_error($ch);
            curl_close($ch);

            // هندلینگ خطای قطع شدن اینترنت سرور
            if ($err) {
                // اگر تایم‌اوت خورد، یک بار دیگه تلاش کن شاید شبکه کند بوده
                if ($attempt <= $maxRetries) { 
                    sleep(2); 
                    continue; 
                }
                throw new Exception("Connection Error (Timeout): $err");
            }

            $decoded = json_decode($response, true);
            
            // موفقیت
            if ($httpCode === 200) {
                return $decoded;
            }

            // هندلینگ خطای سهمیه یا شلوغی سرور گوگل
            $msg = $decoded['error']['message'] ?? "Unknown API Error";
            
            if ($httpCode === 429 || $httpCode === 503) {
                // صبر نمایی (Exponential Backoff)
                sleep(5 * $attempt);
                continue;
            }
            
            // اگر ارور 400 یا 404 بود دیگر تلاش مجدد فایده ندارد
            throw new Exception("API Error ($httpCode): $msg");
        }
        
        throw new Exception("Max retries exceeded. Service unavailable.");
    }
}