<?php

namespace App\Services\Agents;

use App\Services\AI\GeminiClient;

class WriterAgent
{
    private GeminiClient $client;

    public function __construct(GeminiClient $client)
    {
        $this->client = $client;
    }

    public function writeReport(string $topic, string $notes, array $ctx = [], string $model = 'gemini-1.5-pro'): string
    {
        // 1. دریافت ورودی‌ها
        $role = $ctx['role'] ?? 'Senior Subject Matter Expert';
        $lang = $ctx['language'] ?? 'Persian';
        $wordCount = $ctx['word_count'] ?? '2500';
        $focusKw = $ctx['focus_keyword'] ?? '';
        
        // این‌ها را حتما دریافت میکنیم تا تنظیمات کاربر در سایدبار نپرد
        $userConstraints = $ctx['constraints'] ?? '';
        $userTone = $ctx['tone'] ?? 'Smart & Relatable'; 
        
        $targetAudience = $ctx['audience'] ?? 'General Audience seeking expert advice';

        // 2. موتور بومی‌سازی داینامیک (Logic by You) - عالیه
        $localization = "";
        if ($lang === 'Persian' || $lang === 'Farsi') {
            $localization = <<<TXT
            ### 🇮🇷 IRANIAN CONTEXT ENGINE (Critical):
            You are writing for an Iranian audience living in Iran. Adapt your examples dynamically based on the topic:
            
            1. **If Product/Tech:** Reference local availability, warranty issues, and brands popular in Iran (Samsung, Xiaomi, Chinese cars, Snapp, Divar).
            2. **If Economy/Finance:** Acknowledge inflation/currency volatility implicitly as a shared struggle.
            3. **If Lifestyle/General:** Use cultural metaphors.
            4. **Voice:** Use "You" (شما). Be a "Compassionate Expert" (متخصص دلسوز). Avoid robot-translated phrasing.
            TXT;
        }

        // 3. سئو
        $seoInstruction = "";
        if (!empty($focusKw)) {
            $seoInstruction = "### 🚀 SEO CORE:\n- Focus Keyword: '**{$focusKw}**'.\n- Strategy: Semantic richness over keyword stuffing.";
        }

        // 4. استایل‌دهی (سبز)
        $visualEvidence = <<<TXT
        ### 🔍 VISUAL AUTHORITY:
        When stating a hard statistic, technical spec, or law of physics, highlight it:
        <span style="background-color: #dcfce7; color: #166534; padding: 0 4px; border-radius: 4px; font-weight: 500;">(Fact) [Insert Fact]</span>
        TXT;

        // 5. ساختار (Master Structure)
        $prompt = <<<EOT
        ACT AS: {$role}.
        AUDIENCE: {$targetAudience}.
        TOPIC: "{$topic}"
        LANGUAGE: {$lang}.
        LENGTH: ~{$wordCount} words.
        TONE: {$userTone}.

        {$seoInstruction}
        {$localization}
        {$visualEvidence}

        ### 🧠 CONTENT ARCHITECTURE (Expert-Storyteller Mode):

        **Phase 1: The Hook (Story & Pain)**
        - **NEVER** start with "What is X?".
        - **START WITH** a sensory scenario, a common costly mistake, or a specific "pain point" Iranian users face.
        - Hook the reader instantly.

        **Phase 2: The Deep Dive (Technical Authority)**
        - Explain core concepts simply but accurately.
        - Use comparisons (A vs B) to clarify.
        - Mention brands/models relevant to the user (e.g. if cars: Dena, 207).

        **Phase 3: The "Hidden Angles" (Inside Secrets)**
        - Discuss what other sites MISS.
        - Talk about external factors (Heat, Dust, Fake Parts, Filtering).
        - Connect specs to daily life benefits.

        **Phase 4: Practical Guide & Solutions**
        - Actionable checklist.
        - Solutions specifically working in Iran.

        ### 🚫 NEGATIVE CONSTRAINTS (Strictly Enforce):
        - {$userConstraints}
        - NO fluff words ("Nowadays", "It is said").
        - NO passive voice.
        - NO "Summary" section; use "Conclusion/Action Plan" instead.

        --------------------------------------------------
        ### 📚 RESEARCH NOTES:
        {$notes}
        --------------------------------------------------

        ### 📝 OUTPUT (Markdown):
        
        ### 1. SEO PACKAGE
        - **Slug:** ...
        - **Meta Title:** ...
        - **Meta Description:** ...

        ### 2. ARTICLE
        # [Magnet Title]

        [Story Intro]

        [Body with H2s, H3s and Green Spans]

        ## سوالات متداول
        (3 Real User Questions)

        ## کلام آخر
        (Expert Verdict)

        [START WRITING]:
        EOT;

        $response = $this->client->generateContent($prompt, $model, false);
        return $response['candidates'][0]['content']['parts'][0]['text'] ?? "Error.";
    }
}