#!/usr/bin/env php
<?php

// ۱. لود کردن اتولودر و تنظیمات
require_once __DIR__ . '/../vendor/autoload.php';

use App\Core\Env;
use App\Models\ResearchJob;
use App\Services\AI\GeminiClient;
use App\Services\Agents\PlannerAgent;
use App\Services\Agents\ResearcherAgent;
use App\Services\Agents\WriterAgent;

// بارگذاری متغیرهای محیطی
Env::load(__DIR__ . '/../.env');

echo "🚀 Research Worker Started [PID: " . getmypid() . "]\n";
echo "Waiting for jobs...\n";

// ۲. راه‌اندازی سرویس‌های هوش مصنوعی (بیرون حلقه برای پرفورمنس)
try {
    $client = new GeminiClient();
    $planner = new PlannerAgent($client);
    $researcher = new ResearcherAgent($client);
    $writer = new WriterAgent($client);
} catch (Exception $e) {
    die("❌ Fatal Error: " . $e->getMessage() . "\n");
}

// ۳. حلقه اصلی (Infinite Loop)
while (true) {
    // گرفتن قدیمی‌ترین کار در انتظار
    try {
        $job = ResearchJob::getNextPendingJob();
    } catch (Exception $e) {
        echo "DB Error: " . $e->getMessage() . "\n";
        sleep(5);
        continue;
    }

    if (!$job) {
        // اگر کاری نبود، ۲ ثانیه استراحت کن تا CPU درگیر نشود
        sleep(2);
        continue;
    }

    $id = $job['id'];
    $topic = $job['user_topic'];

    // --- تغییر جدید: خواندن تنظیمات کاربر (مخاطب، زبان و...) ---
    $userParams = json_decode($job['params'] ?? '{}', true);
    
    // تنظیم پیش‌فرض‌ها برای جلوگیری از ارور
    $context = [
        'audience' => $userParams['audience'] ?? 'General Audience',
        'format'   => $userParams['format'] ?? 'Blog Article',
        'tone'     => $userParams['tone'] ?? 'Professional',
        'language' => $userParams['language'] ?? 'Persian'
    ];
    
    echo "\n----------------------------------------\n";
    echo "💡 Found Job: {$id}\n";
    echo "   Topic: {$topic}\n";
    echo "   Settings: {$context['format']} in {$context['language']} for {$context['audience']}\n";

    try {
        // --- فاز ۱: برنامه‌ریزی (Planner) ---
        echo "   ➜ Phase 1: Planning Strategy...\n";
        ResearchJob::update($id, ['status' => 'planning']);
        
        // ارسال کانتکست به پلنر (تا سوالات را متناسب با مخاطب طرح کند)
        // نکته: مطمئن شوید متد generatePlan در کلاس PlannerAgent ورودی دوم را می‌گیرد
        $plan = $planner->generatePlan($topic, $context);
        
        ResearchJob::update($id, [
            'plan_data' => json_encode($plan)
        ]);
        echo "     ✔ Plan Created (" . count($plan) . " steps)\n";


        // --- فاز ۲: تحقیق و جستجو (Researcher) ---
        echo "   ➜ Phase 2: Researching (Deep Dive)...\n";
        ResearchJob::update($id, ['status' => 'researching']);
        
        $notes = $researcher->research($plan);
        
        ResearchJob::update($id, [
            'research_notes' => $notes
        ]);
        echo "     ✔ Research Completed (" . strlen($notes) . " chars gathered)\n";


        // --- فاز ۳: نگارش (Writer) ---
        echo "   ➜ Phase 3: Writing Report...\n";
        ResearchJob::update($id, ['status' => 'writing']);
        
        // ارسال کانتکست به نویسنده (بسیار مهم برای زبان و لحن)
        // نکته: مطمئن شوید متد writeReport در کلاس WriterAgent ورودی سوم را می‌گیرد
        $finalReport = $writer->writeReport($topic, $notes, $context);
        
        // --- پایان موفقیت‌آمیز ---
        ResearchJob::update($id, [
            'status' => 'completed',
            'final_report' => $finalReport
        ]);
        
        echo "✅ JOB COMPLETED: {$id}\n";

    } catch (Exception $e) {
        // مدیریت خطا: اگر هر جایی به مشکل خورد، وضعیت را Failed کن
        echo "❌ Job Failed: " . $e->getMessage() . "\n";
        // ذخیره متن خطا در صورت امکان برای دیباگ (اختیاری)
        ResearchJob::update($id, ['status' => 'failed']);
    }
}